//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

package ordertracker

import (
	"context"
	"database/sql"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"de.heapoverflow/ordertracker/logger"
)

type OrderTracker struct {
	Logger      *RequestLogger
	DB          *sql.DB
	DocumentDir string
}

type OrderTrackerServer struct {
	ot      *OrderTracker
	server  *http.Server
	baseURL *url.URL
}

func NewOrderTrackerServer(l *logger.Logger, db *sql.DB, listenAddrPort string, documentDir string) (*OrderTrackerServer, error) {
	ot := &OrderTracker{
		Logger:      &RequestLogger{Logger: l},
		DB:          db,
		DocumentDir: documentDir,
	}

	baseURL, err := url.Parse(fmt.Sprintf("http://%s/", listenAddrPort))
	if err != nil {
		return nil, err
	}

	s := &OrderTrackerServer{ot: ot, baseURL: baseURL}

	var handler http.Handler
	handler = s
	handler = RequestLogging(ot.Logger, handler)
	handler = RequestTracing(handler)

	s.server = &http.Server{
		Addr:     listenAddrPort,
		Handler:  handler,
		ErrorLog: l.InfoLogger(),
		// see https://blog.cloudflare.com/exposing-go-on-the-internet/
		ReadTimeout:  5 * time.Second,
		WriteTimeout: 10 * time.Second,
		IdleTimeout:  120 * time.Second,
	}

	if _, err := s.ot.DB.Exec(`
		CREATE TABLE IF NOT EXISTS orders (
			rid INTEGER NOT NULL PRIMARY KEY,
			id BLOB NOT NULL UNIQUE,
			order_no TEXT NOT NULL UNIQUE,
			created_date TIMESTAMP,
			preferred_date TIMESTAMP,
			confirmed_date TIMESTAMP,
			comment TEXT,
			order_document TEXT,
			order_document_hash BLOB,
			demand_plan_document TEXT,
			demand_plan_document_hash BLOB,
			confirmation_document TEXT,
			confirmation_document_hash BLOB
		)`); err != nil {
		return nil, err
	}
	if _, err := s.ot.DB.Exec(`
		CREATE UNIQUE INDEX IF NOT EXISTS idx_orders_id
		ON orders(id)`); err != nil {
		return nil, err
	}
	if _, err := s.ot.DB.Exec(`
		CREATE UNIQUE INDEX IF NOT EXISTS idx_orders_order_no
		ON orders(order_no)`); err != nil {
		return nil, err
	}

	return s, nil
}

func (s *OrderTrackerServer) Run() error {
	return s.server.ListenAndServe()
}

func (s *OrderTrackerServer) Shutdown() error {
	ctx, _ := context.WithTimeout(context.Background(), 5*time.Second)
	s.server.SetKeepAlivesEnabled(false)
	return s.server.Shutdown(ctx)
}

func (s *OrderTrackerServer) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	var handler http.Handler
	prefix, remaining := ShiftPath(r.URL.Path)
	baseURL := AppendURLPath(s.baseURL, prefix)
	switch prefix {
	case "/api":
		prefix, remaining = ShiftPath(remaining)
		baseURL = AppendURLPath(baseURL, prefix)
		switch prefix {
		case "/orders":
			handler = NewOrderHandler(s.ot, baseURL)
			handler = ReplacePath(remaining, handler)
		default:
			handler = http.NotFoundHandler()
		}
	default:
		handler = NewStaticHandler(s.ot)
	}

	handler.ServeHTTP(w, r)
}
